//
//  ContentView.swift
//  SpeedKeys
//
//  Created by Jose Polo Ramos on 2/24/25.
//

import SwiftUI

// Commented out constant - likely used for sidebar button width elsewhere
//let sidebarButtonWidth:CGFloat = 60

/// Simple identifiable structure for grid content
struct GridContent: Identifiable {
    var id = UUID()
}

/// Custom text field component for entering commands
/// Manages bidirectional data binding between the UI and the underlying data model
struct CommandTextField: View {
    
    // Binding to the entire entries collection
    @Binding var data: Entries
    // Binding to the specific entry being edited
    @Binding var item: Entry
    // Local state for the text field value
    @State var text: String
    
    /// Initializer that sets up the text field with existing command data
    /// - Parameters:
    ///   - data: Binding to the entries collection
    ///   - item: Binding to the specific entry
    init(data: Binding<Entries>, item: Binding<Entry>) {
        self._data = data
        self._item = item
        // Initialize text with the existing command or empty string
        text = self._item.wrappedValue.command.isEmpty ? "" : self._item.wrappedValue.command
    }
    
    var body: some View {
        
        TextField(
          "custom command",
          text: $text
        )
        .controlSize(.large)
        .padding(.vertical, 5)
        .frame(height: 25, alignment: .leading)
        .lineLimit(3...5)
        .padding()
        .fixedSize(horizontal: false, vertical: true)
        .padding()
        .frame(width: 850, height: 25, alignment: .leading)
        .onSubmit {
            // Save the command when user submits (presses Enter)
            item.command = text
            // Commented out alternative save method
            // data.put(value: text, asType : .command , forKey : item.key )
        }
    }
}
 
/// Main content view that displays a grid of entry items
/// Each entry shows controls for deletion, shortcuts, file selection, and commands
struct ContentView: View {
    
    // Singleton instance of the entries data model
    @State var data: Entries = Entries.instance
    // Controls modal visibility (currently unused)
    @State private var showModal = false
    // Tracks the currently selected item
    @State var selected: String
    
    // Grid configuration with a single flexible column
    let columns = [
        GridItem(.flexible(minimum: 20, maximum: 3000))
    ]
    
    /// Deletes an entry from the data model by its key
    /// - Parameter key: The unique key identifying the entry to delete
    func deleteItem(key: String) {
        data.delete(entryWithKey: key)
    }
    
    var body: some View {
        VStack {
            // Top action bar for global actions
            ActionBar(data: $data)
            
            // Scrollable grid of entries
            ScrollView {
                LazyVGrid(columns: columns, alignment: .leading, spacing: 16) {
                    // Iterate through all entries with two-way binding
                    ForEach($data.entries, id: \.self) { $item in
                        VStack {
                            HStack(alignment: VerticalAlignment.center) {
                                
                                // Delete button - removes this entry
                                Button(action: { deleteItem(key: item.key) }) {
                                    Label("", systemImage: "minus.circle.fill")
                                        .font(.system(size: 22, weight: .bold))
                                        .foregroundColor(.red)
                                        .frame(width: 20, alignment: .center)
                                }
                                .frame(width: 40, alignment: .center)
                                .padding(10)
                                .buttonStyle(PlainButtonStyle())
                                
                                // Visual separator
                                Divider()
                                    .frame(width: 1, height: 25)
                                    .overlay(Color.gray.opacity(0.3))
                                
                                // Shortcut configuration button
                                ShortCutButton(data: $data, item: $item)
                                
                                Divider()
                                    .frame(width: 1, height: 25)
                                    .overlay(Color.gray.opacity(0.3))
                                
                                // Application file selector
                                FileButtons(data: $data, isApplicationSelected: true, selected: $item)
                                
                                Divider()
                                    .frame(width: 1, height: 25)
                                    .overlay(Color.gray.opacity(0.3))
                                
                                // Non-application file selector
                                FileButtons(data: $data, isApplicationSelected: false, selected: $item)
                                
                                Divider()
                                    .frame(width: 1, height: 25)
                                    .overlay(Color.gray.opacity(0.3))
                                
                                // Command input text field
                                CommandTextField(data: $data, item: $item)
                            }
                        }
                        .frame(maxWidth: .infinity, maxHeight: .infinity)
                        .padding(.top)
                        .background(Color.black)
                        .clipShape(RoundedRectangle(cornerRadius: 5))
                    }
                }
                .padding(.horizontal)
            }
        }
    }
}

#Preview {
    ContentView(selected: "")
}
